import type { Category, RestaurantInfo, Product, Addon } from './types';
import { v4 as uuidv4 } from 'uuid';


// Default data. This will be used if nothing is in localStorage.
export const defaultRestaurantInfo: RestaurantInfo = {
  name: 'MenuZap Fast',
  logoUrl: 'https://images.unsplash.com/photo-1760037028517-e5cc6e3ebd3e?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHwzfHxsb2dvJTIwYWJzdHJhY3R8ZW58MHx8fHwxNzYzODQwMTA4fDA&ixlib=rb-4.1.0&q=80&w=1080',
  minOrder: 20.0,
  city: 'São Paulo',
  deliveryTime: '30-50 min',
  whatsappNumber: '5511999999999', // Replace with a real number. Example: 5511912345678
  webhookUrl: '',
  facebookPixelHead: '',
  facebookPixelBody: '',
  googleTagManagerHead: '',
  googleTagManagerBody: '',
  googleAnalyticsHead: '',
  googleAnalyticsBody: '',
  mercadoPagoPublicKey: '',
  footerText: 'Todos os direitos reservados.',
  carouselImages: [
    {
      "id": "carousel-1",
      "description": "Delicious pizza being prepared",
      "imageUrl": "https://images.unsplash.com/photo-1532460734809-e7f8475ca917?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHwxfHxwaXp6YSUyMHByZXBhcmF0aW9ufGVufDB8fHx8MTc2Mzg4MTQ2MHww&ixlib=rb-4.1.0&q=80&w=1080",
      "imageHint": "pizza preparation"
    },
    {
      "id": "carousel-2",
      "description": "A juicy and delicious burger",
      "imageUrl": "https://images.unsplash.com/photo-1625331725309-83e4f3c1373b?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHwzfHxkZWxpY2lvdXMlMjBidXJnZXJ8ZW58MHx8fHwxNzYzODgxNDYwfDA&ixlib=rb-4.1.0&q=80&w=1080",
      "imageHint": "delicious burger"
    },
    {
      "id": "carousel-3",
      "description": "A refreshing and colorful drink",
      "imageUrl": "https://images.unsplash.com/photo-1634825881542-9bd54ca437ea?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHw1fHxjb2xvcmZ1bCUyMGRyaW5rfGVufDB8fHx8MTc2Mzc4Njg5OXww&ixlib=rb-4.1.0&q=80&w=1080",
      "imageHint": "colorful drink"
    },
    {
      "id": "carousel-4",
      "description": "A tempting and sweet dessert",
      "imageUrl": "https://images.unsplash.com/photo-1710106519622-8c49d0bcff2f?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHwxfHxzd2VldCUyMGRlc3NlcnR8ZW58MHx8fHwxNzYzODE4NjEzfDA&ixlib=rb-4.1.0&q=80&w=1080",
      "imageHint": "sweet dessert"
    }
  ],
  colors: {
    primary: '20 100% 66%',
    background: '34 78% 91%',
    accent: '48 100% 67%',
    foreground: '20 10% 20%',
    card: '34 78% 95%',
    cardForeground: '20 10% 20%',
    mutedForeground: '0 0% 45.1%',
  },
};

export let categories: Category[] = [
  {
    id: 'pizzas',
    name: 'Pizzas',
    products: [
      {
        id: 'pizz-1',
        name: 'Mussarela',
        description: 'Molho de tomate fresco, mussarela de alta qualidade e orégano.',
        price: 40.0,
        imageUrl: 'https://images.unsplash.com/photo-1692737580558-b9dfdac5599c?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHw3fHxwaXp6YSUyMGNoZWVzZXxlbnwwfHx8fDE3NjM4NzA5OTR8MA&ixlib=rb-4.1.0&q=80&w=1080',
        addons: [
          { id: 'addon-borda', name: 'Borda Recheada Catupiry', price: 8.0 },
          { id: 'addon-bacon', name: 'Extra Bacon', price: 6.0 },
        ],
      },
      {
        id: 'pizz-2',
        name: 'Calabresa',
        description: 'Molho de tomate, mussarela, calabresa fatiada e cebola.',
        price: 42.5,
        imageUrl: 'https://images.unsplash.com/photo-1564128442383-9201fcc740eb?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHw0fHxwaXp6YSUyMHBlcHBlcm9uaXxlbnwwfHx8fDE3NjM4Mzg4ODB8MA&ixlib=rb-4.1.0&q=80&w=1080',
        addons: [
          { id: 'addon-borda', name: 'Borda Recheada Catupiry', price: 8.0 },
          { id: 'addon-cheddar', name: 'Extra Cheddar', price: 5.0 },
        ],
      },
      {
        id: 'pizz-3',
        name: 'Portuguesa',
        description: 'Molho, mussarela, presunto, ovo, cebola, pimentão e azeitonas.',
        price: 45.0,
        imageUrl: 'https://images.unsplash.com/photo-1507273026339-31b655f3752d?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHw0fHxwaXp6YSUyMHZlZ2V0YWJsZXxlbnwwfHx8fDE3NjM4NzA5OTR8MA&ixlib=rb-4.1.0&q=80&w=1080',
        addons: [
          { id: 'addon-borda', name: 'Borda Recheada Catupiry', price: 8.0 },
        ],
      },
      {
        id: 'pizz-4',
        name: 'Frango com Catupiry',
        description: 'Molho de tomate, frango desfiado coberto com catupiry cremoso.',
        price: 48.0,
        imageUrl: 'https://images.unsplash.com/photo-1604917869287-3ae73c77e227?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHwxMHx8cGl6emElMjBjaGlja2VufGVufDB8fHx8MTc2Mzg3MDk5NHww&ixlib=rb-4.1.0&q=80&w=1080',
        addons: [
          { id: 'addon-borda', name: 'Borda Recheada Catupiry', price: 8.0 },
          { id: 'addon-milho', name: 'Extra Milho', price: 3.0 },
        ],
      },
    ],
  },
  {
    id: 'bebidas',
    name: 'Bebidas',
    products: [
      {
        id: 'drink-1',
        name: 'Coca-Cola',
        description: 'Lata 350ml',
        price: 5.0,
        imageUrl: 'https://images.unsplash.com/photo-1585498154575-3db0fda49f1d?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHw1fHxzb2RhJTIwY2FufGVufDB8fHx8MTc2Mzg2MzMwN3ww&ixlib=rb-4.1.0&q=80&w=1080',
        addons: [],
      },
      {
        id: 'drink-2',
        name: 'Guaraná Antarctica',
        description: 'Lata 350ml',
        price: 5.0,
        imageUrl: 'https://images.unsplash.com/photo-1603569239784-0a4b159fa3dc?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHw3fHxzb2RhJTIwY2FufGVufDB8fHx8MTc2Mzg2MzMwN3ww&ixlib=rb-4.1.0&q=80&w=1080',
        addons: [],
      },
      {
        id: 'drink-3',
        name: 'Suco de Laranja Natural',
        description: 'Copo 500ml',
        price: 8.0,
        imageUrl: 'https://images.unsplash.com/photo-1650460069032-3c410224fe55?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHw3fHxvcmFuZ2UlMjBqdWljZXxlbnwwfHx8fDE3NjM4NzA5OTR8MA&ixlib=rb-4.1.0&q=80&w=1080',
        addons: [],
      },
    ],
  },
  {
    id: 'lanches',
    name: 'Lanches',
    products: [
      {
        id: 'sand-1',
        name: 'X-Burger',
        description: 'Pão, hambúrguer, queijo, alface, tomate e maionese da casa.',
        price: 25.0,
        imageUrl: 'https://images.unsplash.com/photo-1568901346375-23c9450c58cd?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHwyfHxjaGVlc2VidXJnZXJ8ZW58MHx8fHwxNzYzODcwOTk0fDA&ixlib=rb-4.1.0&q=80&w=1080',
        addons: [
          { id: 'addon-bacon', name: 'Extra Bacon', price: 6.0 },
          { id: 'addon-ovo', name: 'Extra Ovo', price: 3.0 },
        ],
      },
      {
        id: 'sand-2',
        name: 'X-Bacon',
        description: 'Pão, hambúrguer, queijo, fatias de bacon crocante, alface e tomate.',
        price: 28.5,
        imageUrl: 'https://images.unsplash.com/photo-1619810816144-68dbc1f695e8?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHw1fHxiYWNvbiUyMGJ1cmdlcnxlbnwwfHx8fDE3NjM4NjMzMDd8MA&ixlib=rb-4.1.0&q=80&w=1080',
        addons: [
          { id: 'addon-cheddar', name: 'Extra Cheddar', price: 5.0 },
           { id: 'addon-ovo', name: 'Extra Ovo', price: 3.0 },
        ],
      },
    ],
  },
  {
    id: 'sobremesas',
    name: 'Sobremesas',
    products: [
      {
        id: 'dessert-1',
        name: 'Pudim de Leite',
        description: 'Uma fatia generosa do nosso pudim caseiro.',
        price: 10.0,
        imageUrl: 'https://images.unsplash.com/photo-1605256107786-c598074d5027?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHwxfHxwdWRkaW5nfGVufDB8fHx8MTc2MzgzODg4MHww&ixlib=rb-4.1.0&q=80&w=1080',
        addons: [],
      },
      {
        id: 'dessert-2',
        name: 'Brownie com Sorvete',
        description: 'Brownie quentinho com uma bola de sorvete de creme.',
        price: 18.0,
        imageUrl: 'https://images.unsplash.com/photo-1702827402870-7c33dc7b67be?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHwxfHxicm93bmllJTIwaWNlY3JlYW18ZW58MHx8fHwxNzYzODcwOTk0fDA&ixlib=rb-4.1.0&q=80&w=1080',
         addons: [
          { id: 'addon-calda', name: 'Calda de Chocolate', price: 4.0 },
        ],
      },
    ],
  },
];

export function addCategory(name: string) {
    if (!name) return;
    const newCategory: Category = {
        id: uuidv4(),
        name,
        products: [],
    };
    categories.unshift(newCategory);
    return newCategory;
}

export function removeCategory(categoryId: string) {
    categories = categories.filter(c => c.id !== categoryId);
}

export function addProduct(categoryId: string, product: Omit<Product, 'id' | 'imageUrl'>) {
    const category = categories.find(c => c.id === categoryId);
    if (!category) {
        console.error(`Category ${categoryId} not found.`);
        return;
    }
    const newProduct: Product = {
        ...product,
        id: `${categoryId.slice(0,4)}-${category.products.length + 1}`,
        imageUrl: 'https://images.unsplash.com/photo-1649494229954-85b802e24d7e?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3NDE5ODJ8MHwxfHNlYXJjaHw0fHxmb29kJTIwZGVsaWNpb3VzfGVufDB8fHx8MTc2MzgwNDEwOHww&ixlib=rb-4.1.0&q=80&w=1080',
        addons: [],
    }
    category.products.push(newProduct);
    return newProduct;
}

export function removeProduct(categoryId: string, productId: string) {
    const category = categories.find(c => c.id === categoryId);
    if (!category) {
        console.error(`Category ${categoryId} not found.`);
        return;
    }
    category.products = category.products.filter(p => p.id !== productId);
}

export function updateProduct(categoryId: string, productId: string, updatedData: Partial<Omit<Product, 'id' | 'addons'>>) {
    const category = categories.find(c => c.id === categoryId);
    if (!category) return;
    const productIndex = category.products.findIndex(p => p.id === productId);
    if (productIndex === -1) return;

    category.products[productIndex] = {
        ...category.products[productIndex],
        ...updatedData
    };
    return category.products[productIndex];
}

export function addAddonToProduct(categoryId: string, productId: string, addon: Omit<Addon, 'id'>) {
    const category = categories.find(c => c.id === categoryId);
    if (!category) return;
    const product = category.products.find(p => p.id === productId);
    if (!product) return;

    if (!product.addons) {
        product.addons = [];
    }

    const newAddon: Addon = {
        ...addon,
        id: `addon-${product.id}-${product.addons.length + 1}-${Date.now()}`
    };

    product.addons.push(newAddon);
    return newAddon;
}

export function removeAddonFromProduct(categoryId: string, productId: string, addonId: string) {
    const category = categories.find(c => c.id === categoryId);
    if (!category) return;
    const product = category.products.find(p => p.id === productId);
    if (!product || !product.addons) return;

    product.addons = product.addons.filter(a => a.id !== addonId);
}
